
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1008.C - Message 1008 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1008 messages.
//
//
//   REVISION HISTORY
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <graph.h>
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"
#include "labmon.h"
#include "util.h" 
#include "display.h"

extern tBOOL DisplayDRError;

tDOUBLE utrip2dbl (tUTRIPLE *x);    

////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1008 message from the ZMsgBuf.
//      
void Dec1008(tMSGBUF *ZMsgBuf, tMSG1008 *Msg)
{  
   tSHORT i;

   GetShort  (&Msg->SequenceNumber     , ZMsgBuf,  8);     // ND  
   GetUShort (&Msg->GPSWeek            , ZMsgBuf,  9);     // ND
   GetULong  (&Msg->GPSTimeSeconds     , ZMsgBuf, 10);     // ND
   GetULong  (&Msg->GPSTimeNanoseconds , ZMsgBuf, 12);     // ND
   GetUShort (&Msg->NumSatsUsed        , ZMsgBuf, 14);     // ND
   GetUShort (&Msg->GDOP               , ZMsgBuf, 15);     // 1E+02 * ND
   GetUShort (&Msg->PDOP               , ZMsgBuf, 16);     // 1E+02 * ND
   GetUShort (&Msg->HDOP               , ZMsgBuf, 17);     // 1E+02 * ND
   GetUShort (&Msg->VDOP               , ZMsgBuf, 18);     // 1E+02 * ND
   GetUShort (&Msg->TDOP               , ZMsgBuf, 19);     // 1E+02 * ND
                              
   for(i=0; i<GI_NUM_HW_CHAN; i++)  {  
     // perform special decoding to unpack words
     GetBit (&BestChanSum[i].BestChanStatus.MeasurementUsed,       ZMsgBuf, 20+10*i,  0);  // 1=true
     GetBit (&BestChanSum[i].BestChanStatus.EphemerisAvailable,    ZMsgBuf, 20+10*i,  1);  // 1=true
     GetBit (&BestChanSum[i].BestChanStatus.DifferentialAvailable, ZMsgBuf, 20+10*i,  2);  // 1=true
     GetBit (&BestChanSum[i].BestChanStatus.MeasurementValid,      ZMsgBuf, 20+10*i,  3);  // 1=true
     GetBits(&BestChanSum[i].BestChanStatus.CN0,                   ZMsgBuf, 20+10*i,  4, 6);  // ND
     GetBits(&BestChanSum[i].BestChanStatus.PRN,                   ZMsgBuf, 20+10*i, 10, 6);  // ND
                                                               
     GetUTriple (&BestChanSum[i].CodePhase.ThreeShorts,    ZMsgBuf, 21+10*i); // ND
     GetUTriple (&BestChanSum[i].CarrierPhase.ThreeShorts, ZMsgBuf, 24+10*i); // ND
     GetLong    (&BestChanSum[i].CarrierVelocity,          ZMsgBuf, 27+10*i); // ND
     GetUShort  (&BestChanSum[i].PhaseBiasCount,           ZMsgBuf, 29+10*i); // ND
   }                                              
   GetUShort  (&Msg->GPSHeadingError,	ZMsgBuf, 140); // ND
   GetUShort  (&Msg->GPSSpeedError,		ZMsgBuf, 141); // ND
   GetULong   (&Msg->GPSPositionError,	ZMsgBuf, 142); // ND
   GetUShort  (&Msg->DRHeadingError,	ZMsgBuf, 144); // ND
   GetUShort  (&Msg->DRSpeedError,		ZMsgBuf, 145); // ND
   GetULong   (&Msg->DRPositionError,	ZMsgBuf, 146); // ND
}                                      

////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1008 message in the ZMsgBuf. 
//
void Bld1008(tMSGBUF *ZMsgBuf, tMSG1008 *Msg)
{
   // ack flags assumed preset

   PutShort  (0x81FF                       , ZMsgBuf, 1);         // message preamble
   PutShort  (1008                         , ZMsgBuf, 2);         // message number
   PutShort  (0                            , ZMsgBuf, 3);         // message length 
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1008 message from the ZMsgBuf.
//      
void Show1008(tMSG1008 *Msg)
{  
   char    Buf[80] = ""; 
//   tSHORT  i;

   char    *DayName[7] ={"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"}; 
//   char    DayOfWeek[5]; 
//   double  GPSTimeSeconds,GPSTimeNSeconds;
//   unsigned long  tempsecs;
   unsigned short gpsprecision = 7;  // used to calculate precision field of gpssec output
   
//   double  CodePhase    [GI_NUM_HW_CHAN];
//   double  CarrierPhase [GI_NUM_HW_CHAN];
//   double  CarrierRate  [GI_NUM_HW_CHAN];

   // convert values  
/*
   // calculate gpssec field
   tempsecs = Msg->GPSTimeSeconds;
   while( tempsecs /= 10 ){ gpsprecision--; };
   // get rid of roundoff for nano seconds
   GPSTimeNSeconds = (double) ( Msg->GPSTimeNanoseconds / (unsigned long) pow( 10.0, (9.0 - (double) gpsprecision) ) );
   GPSTimeNSeconds *= pow( 10.0, (9.0 - (double) gpsprecision) );
   GPSTimeSeconds = Msg->GPSTimeSeconds + GPSTimeNSeconds / 1E+09;

   sprintf(DayOfWeek,"%s",
           DayName[(short)(Msg->GPSTimeSeconds / 86400)]);

   // get RNG data
   for(i=0; i<GI_NUM_HW_CHAN; i++){
     
      CodePhase[i]    = utrip2dbl(&BestChanSum[i].CodePhase.ThreeShorts) 
                               * GF_SPEED_OF_LIGHT / (X50TWOP45); 
      CarrierPhase[i] = utrip2dbl(&BestChanSum[i].CarrierPhase.ThreeShorts) 
                               * GF_SPEED_OF_LIGHT / (X50TWOP45);
      CarrierRate[i]  = (BestChanSum[i].CarrierVelocity) 
                               * GF_SPEED_OF_LIGHT / TWOP45;
   }

                                                   
   // output the data values
   sprintf(Buf,"%05d"  ,Msg->SequenceNumber        );ShowText(Buf,TIMR+ 6,TIMC+ 4);
   sprintf(Buf,"%4d "  ,Msg->GPSWeek               );ShowText(Buf,TIMR+ 4,TIMC+ 5);
   sprintf(Buf,"%9.*f" ,gpsprecision,GPSTimeSeconds);ShowText(Buf,TIMR+ 2,TIMC+ 8);
   sprintf(Buf,"%09ld" ,Msg->GPSTimeNanoseconds    );ShowText(Buf,TIMR+ 3,TIMC+ 8);
   sprintf(Buf,"%4d"  ,Msg->NumSatsUsed           ); ShowText(Buf,DOPR- 2,DOPC+ 6);
   sprintf(Buf,"%6.2f" ,Msg->GDOP             / 1E+02);ShowText(Buf,DOPR+ 1,DOPC+ 4);
   sprintf(Buf,"%6.2f" ,Msg->PDOP             / 1E+02);ShowText(Buf,DOPR+ 2,DOPC+ 4);
   sprintf(Buf,"%6.2f" ,Msg->HDOP             / 1E+02);ShowText(Buf,DOPR+ 3,DOPC+ 4);
   sprintf(Buf,"%6.2f" ,Msg->VDOP             / 1E+02);ShowText(Buf,DOPR+ 4,DOPC+ 4);
   sprintf(Buf,"%6.2f" ,Msg->TDOP             / 1E+02);ShowText(Buf,DOPR+ 5,DOPC+ 4);
                 
   // channel data
   for(i=0; i<GI_NUM_HW_CHAN; i++){ 
   sprintf(Buf,"%2d"  ,i + 1                             );ShowText(Buf,CHNR+ i,CHNC   ); 
   sprintf(Buf,"%2d"  ,BestChanSum[i].BestChanStatus.CN0 );ShowText(Buf,CHNR+ i,CHNC+14); 
   sprintf(Buf,"%2d"  ,BestChanSum[i].BestChanStatus.PRN );ShowText(Buf,CHNR+ i,CHNC+ 4);  

   // special processing for output
   sprintf(Buf,"%1d" ,BestChanSum[i].BestChanStatus.MeasurementUsed      ); ShowText(Buf,CHNR+ i,CHNC+ 8);      
   sprintf(Buf,"%1d" ,BestChanSum[i].BestChanStatus.EphemerisAvailable   ); ShowText(Buf,CHNR+ i,CHNC+10);
   sprintf(Buf,"%1d" ,BestChanSum[i].BestChanStatus.MeasurementValid     ); ShowText(Buf,CHNR+ i,CHNC+ 9);
   sprintf(Buf,"%1d" ,BestChanSum[i].BestChanStatus.DifferentialAvailable); ShowText(Buf,CHNR+ i,CHNC+11); 
   
   if(BestChanSum[i].CarrierVelocity != 0){ 
     sprintf(Buf,"%12.3lf",CarrierPhase[i]         );ShowText(Buf,CPMR+ i,CPMC-1   ); 
     sprintf(Buf,"%12.3lf",CodePhase[i]            );ShowText(Buf,PRMR+ i,PRMC-2   ); 
     sprintf(Buf,"%9.3lf" ,CarrierRate[i]          );ShowText(Buf,PRMR+ i,PRMC+11); 
   } else {
     sprintf(Buf,"            "                    );ShowText(Buf,CPMR+ i,CPMC-1   ); 
     sprintf(Buf,"            "                    );ShowText(Buf,PRMR+ i,PRMC-2   ); 
     sprintf(Buf,"         "                       );ShowText(Buf,PRMR+ i,PRMC+11);  
   }

   } // end for
*/
   if (DisplayDRError) { // display GPS/DR error separately
     // output the data items   
     sprintf(Buf,"%9.2G",Msg->GPSHeadingError  /1E+02);ShowText(Buf,POSR   ,POSC+ 5);
     sprintf(Buf,"%9.2G",Msg->GPSSpeedError    /1E+02);ShowText(Buf,POSR+ 1,POSC+ 5);
     sprintf(Buf,"%9.2G",Msg->GPSPositionError /1E+02);ShowText(Buf,POSR+ 2,POSC+ 5);  
     sprintf(Buf,"%9.2G",Msg->DRHeadingError   /1E+02);ShowText(Buf,VELR   ,VELC+ 2); 
     sprintf(Buf,"%9.2G",Msg->DRSpeedError     /1E+02);ShowText(Buf,VELR+ 1,VELC+ 2); 
     sprintf(Buf,"%9.2G",Msg->DRPositionError  /1E+02);ShowText(Buf,VELR+ 2,VELC+ 2); 
   }

}
